package com.agilex.healthcare.directscheduling.integration;

import com.agilex.healthcare.directscheduling.exception.AppointmentException;
import com.agilex.healthcare.directscheduling.exception.CancelAppointmentException;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.jaxrs.json.JacksonJaxbJsonProvider;
import com.fasterxml.jackson.jaxrs.json.JacksonJsonProvider;
import com.fasterxml.jackson.module.jaxb.JaxbAnnotationIntrospector;
import gov.va.vamf.scheduling.direct.domain.ValidationError;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.springframework.stereotype.Service;

import javax.ws.rs.BadRequestException;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.client.*;
import javax.ws.rs.client.Invocation.Builder;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.URI;
import java.util.List;
import java.util.Map;

@Service
public class HttpClientService<T> {

    private static final Log logger = LogFactory.getLog(HttpClientService.class);

    public List<T> getDetails(URI uri, final Class<T> clazz) {
    	return this.getDetails(uri, clazz, null, null);
    }

    public List<T> getDetails(URI uri, final Class<T> clazz, Map<String, String> queryParams) {
    	return this.getDetails(uri, clazz, null, queryParams);
    }

	public List<T> getDetails(URI uri, final Class<T> clazz, Map<String, String> headerParams,  Map<String, String> queryParams) {

		List<T> items = null;
		ParameterizedType genericType = getParameterizedType(clazz);
		final JacksonJsonProvider jacksonJsonProvider = jacksonMapper();
		try {

			WebTarget webTarget = getRestClient().target(uri);
			webTarget = setQueryParams(webTarget, queryParams);
			webTarget.register(JacksonFeature.class);
			webTarget.register(jacksonJsonProvider);
			Builder builder = webTarget.request();
			builder = setHeaderParams(builder, headerParams);
			items =  builder.accept(MediaType.APPLICATION_JSON).get(new GenericType<List<T>>(genericType) {});

		}catch (Exception ex) {
			logger.error("Runttime exception:", ex);
//			throw ex;
		}
		return items;
	}

	public T get(URI uri, final Class<T> clazz, Map<String, String> headerParams,  Map<String, String> queryParams)  {

		T item = null;
		try {

			WebTarget webTarget = getRestClient().target(uri);
			webTarget = setQueryParams(webTarget, queryParams);
			Builder builder = webTarget.request();
			builder = setHeaderParams(builder, headerParams);
			item =  builder.accept(MediaType.APPLICATION_JSON).get(clazz);

		} catch(ResponseProcessingException exception) {
			logger.error("Response Processing Exception  message:", exception);
			return item;
		} catch (BadRequestException exception) {

			logger.error("Bad Request Exception:", exception);
			Response response = exception.getResponse();
			String message = exception.getMessage();
			if (response != null && response.hasEntity()) {
				message = response.readEntity(String.class);
			}
			String errorField = Integer.toString(Response.Status.BAD_REQUEST.getStatusCode());
			throw new AppointmentException(new ValidationError(errorField, message));
		} catch (WebApplicationException exception) {

			logger.error("runtime exception message:", exception);
			String errorField = Integer.toString(exception.getResponse().getStatus());
			throw new AppointmentException(new ValidationError(errorField, "VVS Service : " + exception.getMessage()));
		}
		return item;
	}


	public T putCall(URI uri, final Class<T> clazz, Map<String, String> headerParams,  Object entity) {

		T item = null;
		final JacksonJsonProvider jacksonJsonProvider = jacksonMapper();
		try {

			WebTarget webTarget = getRestClient().target(uri);
			webTarget.register(JacksonFeature.class);
			webTarget.register(jacksonJsonProvider);
			Builder builder = webTarget.request();
			builder = setHeaderParams(builder, headerParams);
			item =  builder.accept(MediaType.APPLICATION_JSON).put(Entity.entity(entity, MediaType.APPLICATION_JSON), clazz);

		} catch(ResponseProcessingException exception) {
			logger.error("Response Processing Exception message:", exception);
			return item;
		} catch (BadRequestException exception) {

			logger.error("Bad Request runtime exception message:", exception);
			Response response = exception.getResponse();
			String message = exception.getMessage();
			if (response != null && response.hasEntity()) {
				message = response.readEntity(String.class);
			}
			String errorField = Integer.toString(Response.Status.BAD_REQUEST.getStatusCode());
			throw new CancelAppointmentException(new ValidationError(errorField, message));
		} catch (WebApplicationException exception) {

			logger.error("Web Application runtime exception message:", exception);
			String errorField = Integer.toString(exception.getResponse().getStatus());
			throw new CancelAppointmentException(new ValidationError(errorField, "VVS Service : " + exception.getMessage()));
		}
		return item;
	}

	public T postCall(URI uri, final Class<T> clazz, Map<String, String> headerParams,  Object entity) {

		T item = null;
		final JacksonJsonProvider jacksonJsonProvider = jacksonMapper();
		try {

			WebTarget webTarget = getRestClient().target(uri);
			webTarget.register(JacksonFeature.class);
			webTarget.register(jacksonJsonProvider);
			Builder builder = webTarget.request();
			builder = setHeaderParams(builder, headerParams);
			item =  builder.accept(MediaType.APPLICATION_JSON).post(Entity.entity(entity, MediaType.APPLICATION_JSON), clazz);

		}  catch(ResponseProcessingException exception) {
			logger.error("Response Processing Exception  message:", exception);
			return item;
		} catch (BadRequestException exception) {

			logger.error("Bad Request Exception:", exception);
			Response response = exception.getResponse();
			String message = exception.getMessage();
			if (response != null && response.hasEntity()) {
				message = response.readEntity(String.class);
			}
			String errorField = Integer.toString(Response.Status.BAD_REQUEST.getStatusCode());
			throw new AppointmentException(new ValidationError(errorField, message));
		} catch (WebApplicationException exception) {

			logger.error("runtime exception message:", exception);
			String errorField = Integer.toString(exception.getResponse().getStatus());
			throw new AppointmentException(new ValidationError(errorField, "VVS Service : " + exception.getMessage()));
		}
		return item;
	}


	private Client getRestClient() {
		return ClientBuilder.newBuilder().build();
	}

	private WebTarget setQueryParams(WebTarget webTarget, Map<String, String> queryParams) {

		if (queryParams == null || queryParams.isEmpty()) {
			return webTarget;
		}

		for (String key : queryParams.keySet()) {
			String value = queryParams.get(key);
			webTarget = webTarget.queryParam(key, value);
		}
		return webTarget;
	}

	private Builder setHeaderParams(Builder builder, Map<String, String> headerParams) {

		if (headerParams == null || headerParams.isEmpty()) {
			return builder;
		}

		for (String headerParamKey : headerParams.keySet()) {
			builder = builder.header(headerParamKey, headerParams.get(headerParamKey));
		}
		return builder;
	}

	private ParameterizedType getParameterizedType(final Class<T> clazz) {

		 ParameterizedType genericType = new ParameterizedType() {

	         public Type[] getActualTypeArguments() {
	             return new Type[] {clazz};
	         }
	         public Type getRawType() {
	             return List.class;
	         }
	         public Type getOwnerType() {
	             return List.class;
	         }
	     };
	     return genericType;
	}

	private JacksonJsonProvider jacksonMapper() {

		ObjectMapper objectMapper = new ObjectMapper();
		objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
		objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
	    objectMapper.setAnnotationIntrospector(new JaxbAnnotationIntrospector(objectMapper.getTypeFactory()));
		final JacksonJsonProvider jacksonJsonProvider = new JacksonJaxbJsonProvider(objectMapper, JacksonJaxbJsonProvider.DEFAULT_ANNOTATIONS)
													.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
													.configure(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY,true);
		return jacksonJsonProvider;
	}
}
